import React from "react";

import { generateMetadataObject } from "@/lib/shared/metadata";
import { safeImport } from "@/utils/safeImport";
import { Metadata } from "next";
import { Inter } from "next/font/google";

import fetchContentType from "@/lib/strapi/fetchContentType";
import { cn } from "@/lib/utils";
import { Navigation } from "@/types/types";
import { toKebabCase } from "@/utils/toKebabCase";
import { ViewTransitions } from "next-view-transitions";

const inter = Inter({
  subsets: ["latin"],
  display: "swap",
  weight: ["400", "500", "600", "700", "800", "900"],
});

export async function generateMetadata({
  params,
}: {
  params: { locale: string; slug: string };
}): Promise<Metadata> {
  const pageData = await fetchContentType(
    "global",
    `&filters[locale][$eq]=${params.locale}&populate=seo.metaImage`,
    true
  );

  const seo = pageData?.seo;
  const metadata = generateMetadataObject(seo);
  return metadata;
}

export default async function LocaleLayout({
  children,
  params: { locale },
}: {
  children: React.ReactNode;
  params: { locale: string };
}) {
  const pageData = await fetchContentType(
    "global",
    `filters[locale][$eq]=${locale}`,
    true
  );

  const navigation = await fetchContentType(
    "navigation",
    `filters[locale][$eq]=${locale}`,
    false
  );

  const getNavigationMenus = async (slug: string) => {
    const navItem = navigation?.find((item: Navigation) => item.slug === slug);
    if (!navItem) return null;
    return fetchContentType(`navigation/render/${navItem.slug}`, "", false);
  };

  const headerNavigationMenus = await getNavigationMenus("header-navigation");
  const footerNavigationMenus = await getNavigationMenus("footer-navigation");

  const HeaderComponent =
    pageData?.header?.cmp_variant &&
    (await safeImport(toKebabCase(pageData?.header?.cmp_variant)));

  const FooterComponent =
    pageData?.footer?.cmp_variant &&
    (await safeImport(toKebabCase(pageData?.footer?.cmp_variant)));

  return (
    <html lang={locale}>
      <ViewTransitions>
        <body
          className={cn(
            inter.className,
            "bg-charcoal antialiased h-full w-full"
          )}
        >
          {HeaderComponent && (
            <HeaderComponent
              navigation={headerNavigationMenus}
              global={pageData.data}
            />
          )}
          {children}
          {FooterComponent && (
            <FooterComponent
              navigation={footerNavigationMenus}
              global={pageData.data}
            />
          )}
        </body>
      </ViewTransitions>
    </html>
  );
}
